﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NumLib.Classes
{


    /// <summary>Matrix whose components are real numbers.</summary>
    public class Matrix : Matrix<double>
    {

        #region Initialization

        protected Matrix() { }

        public Matrix(int dim1, int dim2)
        {
            Init(dim1, dim2);
        }

        public Matrix(int dim1, int dim2, double comp)
        {
            Init(dim1, dim2, comp);
        }

        public Matrix(double[,] array)
        {
            Init(array);
        }

        public Matrix(Matrix cv)
        {
            Init(cv);
        }

        #endregion Initialization


        #region Operations


        // Implementacija podedovanih abstraktnih metod:

        public override double Zero
        { get { return 0.0; } }

        public override double Add(double a, double b)
        { return a + b; }

        public override double Subtract(double a, double b)
        { return a - b; }

        public override double Multiply(double a, double b)
        { return a * b; }

        public override double Divide(double a, double b)
        { return a / b; }

        public override double Negative(double a)
        { return -a; }

        public override double Inverse(double a)
        { return 1.0 / a; }

        public override double Conjugate(double a)
        { return a; }


        // Overloaded operators:

        public static Matrix operator +(Matrix a, Matrix b)
        {
            Matrix ret = new Matrix(a);
            ret.Add(b);
            return ret;
        }

        public static Matrix operator -(Matrix a, Matrix b)
        {
            Matrix ret = new Matrix(a);
            ret.Subtract(b);
            return ret;
        }

        //public static Matrix operator *(Matrix a, Matrix b)
        //{
        //    return a.Multiply(b);
        //}


        public static Matrix operator *(Matrix a, double b)
        {
            Matrix ret = new Matrix(a);
            ret.Multiply(b);
            return ret;
        }

        public static Matrix operator *(double b, Matrix a)
        {
            Matrix ret = new Matrix(a);
            ret.Multiply(b);
            return ret;
        }

        // Produkta dveh matrik nismo mogli definirati v osnovnem razredu, ker produkt nima nujno iste dimenzije kot
        // kateri od operandov. Zato produkt dveh matrik definiramo posebej.

        /// <summary>Multiplication of two matrices.</summary>
        public static Matrix operator *(Matrix a, Matrix b)
        {
            if (a.d1 < 1)
                throw new ArgumentException("Matrix multiplication: first matrix dimension is less than 1.");
            if (a.d2 < 1)
                throw new ArgumentException("Matrix multiplication: second matrix dimension is less than 1.");
            if (a.d2 != b.d1)
                throw new ArgumentException("Matrix multiplication: incompatible dimensions.");
            Matrix ret = new Matrix(a.d1, b.d2);
            for (int i = 0; i < a.d1; ++i)
                for (int j = 0; j < b.d2; ++j)
                {
                    double sum = 0.0;
                    for (int k = 0; k < a.d2; ++k)
                        sum += a[i, k] * b[k, j];
                    ret[i, j] = sum;
                }
            return ret;
        }



        #endregion  // Operations

        public static void Example()
        {
            Console.WriteLine(); Console.WriteLine();
            Console.WriteLine("Example: matrix operations:");
            Matrix A = null, B = null;
            double [,] taba = {{1, 2, 0}, {0, 0, 1}};
            A = new Matrix(taba);
            double [,] tabb = {{1, 2}, {2, 3}, {0, 2}};
            B = new Matrix(tabb);
            Console.WriteLine("Matrix A: " + Environment.NewLine + A.ToString(true));
            Console.WriteLine("Matrix B: " + Environment.NewLine + B.ToString(true));
            Console.WriteLine("A * B = " + (A * B).ToString());

        }

    }  // class Matrix



    /// <summary>Generic matrix class.</summary>
    /// <typeparam name="ComponentType">Type of matrix components.</typeparam>
    public abstract class Matrix<ComponentType>
    {

        #region Contructors_Initialization

        protected Matrix()  // hide default constructor
        {
        }


        public Matrix(int dim1, int dim2, ComponentType comp)
        {
            Init(dim1, dim2, comp);
        }

        public Matrix(ComponentType[,] array)
        {
            Init(array);
        }

        /// <summary>Initializes a Matrix with dimendion dim.</summary>
        /// <param name="dim">Matrix dimension.</param>
        public void Init(int dim1, int dim2)
        {
            if (dim1 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with first dimension less than 1. Specified dimension: " + dim1);
            if (dim2 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with second dimension less than 1. Specified dimension: " + dim2);
            _dim1 = dim1;
            _dim2 = dim2;
            _tab = new ComponentType[dim1,dim2];
            for (int i = 0; i < _dim1; ++i)
                for (int j = 0; j < _dim2; ++j)
                    _tab[i,j] = default(ComponentType);
        }

        /// <summary>Initializes a Matrix with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Matrix dimension.</param>
        public void Init(int dim1, int dim2, ComponentType comp)
        {
            if (dim1 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with first dimension less than 1. Specified dimension: " + dim1);
            if (dim2 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with second dimension less than 1. Specified dimension: " + dim2);
            _dim1 = dim1;
            _dim2 = dim2;
            _tab = new ComponentType[dim1, dim2];
            for (int i = 0; i < _dim1; ++i)
                for (int j = 0; j < _dim2; ++j)
                    _tab[i, j] = comp;
        }

        /// <summary>Initializes a Matrix with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Matrix dimension.</param>
        public void Init(ComponentType[,] array)
        {
            _dim1 = array.GetUpperBound(0)+1;
            _dim2 = array.GetUpperBound(1)+1;
            if (_dim1 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with the first dimension less than 1. Specified dimension: " + _dim1);
            if (_dim2 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with the second dimension less than 1. Specified dimension: " + _dim2);
            _tab = new ComponentType[_dim1, _dim2];
            for (int i = 0; i < _dim1; ++i)
                for (int j = 0; j < _dim2; ++j)
                    _tab[i, j] = array[i,j];
        }

        /// <summary>Initializes a Matrix with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Matrix dimension.</param>
        public void Init(Matrix<ComponentType> m)
        {
            _dim1 = m.d1;
            _dim2 = m.d2;
            if (_dim1 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with the first dimension less than 1. Specified dimension: " + _dim1);
            if (_dim2 < 1)
                throw new IndexOutOfRangeException("Can not create a Matrix with the second dimension less than 1. Specified dimension: " + _dim2);
            _tab = new ComponentType[_dim1,_dim2];
            for (int i = 0; i < _dim1; ++i)
                for (int j = 0; j < _dim2; ++j)
                _tab[i,j] = m[i,j];
        }


        public static ComponentType[,] Copy(Matrix<ComponentType> m)
        {
            ComponentType[,] ret;
            if (m == null)
                ret = null;
            else
            {
                int 
                    dim1 = m.d1,
                    dim2 = m.d2;
                ret = new ComponentType[dim1,dim2];
                for (int i = 0; i < dim1; ++i)
                    for (int j = 0; j < dim2; ++j)
                    ret[i,j] = m[i,j];
            }
            return ret;
        }


        #endregion  // Contructors_Initialization

        #region Data

        protected int _dim1, _dim2;
        protected ComponentType[,] _tab;

        /// <summary>Returns the first Matrix dimension (number of rows).</summary>
        public int Dimension1 { get { return _dim1; } }

        /// <summary>Returns the second Matrix dimension (number of columns).</summary>
        public int Dimension2 { get { return _dim2; } }

        /// <summary>Returns the first Matrix dimension (number of rows).</summary>
        public int d1 { get { return _dim1; } }

        /// <summary>Returns the second Matrix dimension (number of columns).</summary>
        public int d2 { get { return _dim2; } }


        /// <summary>Gets or sets a specific Matrix componene.</summary>
        /// <param name="ind">Component index running from 0 to d-1.</param>
        /// <returns>a referance to a component for a reference type ComponentType or its value for value types.</returns>
        public virtual ComponentType this[int ind1, int ind2]
        {
            get
            {
                if (ind1 < 0 || ind1 >= _dim1)
                    throw new IndexOutOfRangeException("First index " + ind1.ToString() + " is out of range, should be between 0 and "
                        + (_dim1 - 1).ToString() + ".");
                if (ind2 < 0 || ind2 >= _dim2)
                    throw new IndexOutOfRangeException("Secind index " + ind2.ToString() + " is out of range, should be between 0 and "
                        + (_dim2 - 1).ToString() + ".");
                return _tab[ind1,ind2];
            }
            set
            {
                if (ind1 < 0 || ind1 >= _dim1)
                    throw new IndexOutOfRangeException("First index " + ind1.ToString() + " is out of range, should be between 0 and "
                        + (_dim1 - 1).ToString() + ".");
                if (ind2 < 0 || ind2 >= _dim2)
                    throw new IndexOutOfRangeException("Second index " + ind2.ToString() + " is out of range, should be between 0 and "
                        + (_dim2 - 1).ToString() + ".");
                _tab[ind1, ind2] = value;
            }
        }



        #endregion  // Data


        #region Operations


        /// <summary>Redefinition of Tostring(), converts a Matrix to string representation.</summary>
        /// <returns>String representation of a matrix.</returns>
        public override string ToString()
        {
            return ToString(true);
        }

        /// <summary>Converts a matrix to string representation.</summary>
        /// <param name="multiline">If true then a multi-line representation is generated.</param>
        /// <returns></returns>
        public virtual string ToString(bool multiline)
        {
            StringBuilder sb = new StringBuilder();
            sb.Append('{');
            for (int i = 0; i < _dim1; ++i)
            {
                if (multiline)
                    sb.Append(Environment.NewLine + "  ");
                sb.Append("{");
                for (int j = 0; j < _dim2; ++j)
                {
                    sb.Append(this[i,j].ToString());
                    if (j < _dim2 - 1) 
                        sb.Append(", ");
                    else
                        sb.Append("}");
                }
                if (i < _dim1 - 1)
                    sb.Append(", ");
            }
            if (multiline)
                sb.Append(Environment.NewLine);
            sb.Append("}");
            if (multiline)
                sb.Append(Environment.NewLine);
            return sb.ToString();
        }

        // Spodaj deklariramo osnovne operacije na polju, nad katerim je definiran vektor. Te operacije so
        // potrebne, da lahko definiramo seštevanje vektorjev, množenje s skalarjem, skalarni produkt itd.
        // Vendar na tem mestu ne moremo operacij tudi definirati, ker so lahko argumenti poljubnih tipov in
        // zato ni znano vnaprej, kako se dejansko izvede recimo seštevanje ali množenje dveh objektov tega tipa.
        // Da gre samo za deklaracije in ne za definicije, povemo z besedo "abstract". Razred, ki vsebue vsaj 
        // eno abstraktno metodo, mora biti tudi sam deklariran kot abstrakten. Abstraktnega razreda ne moremo
        // instanciirati, lahko pa po njem dedujejo drugi razredi. Razred, ki implementira vse metode, ki so 
        // deklarirane kot abstraktne v abstraktnem razredu, po katerem deduje, je konkreten in ga lahko 
        // instanciiramo (to pomeni, da lahko z operatorjem new() naredimo objekt danega tipa).

        public abstract ComponentType Zero { get; }
        public abstract ComponentType Add(ComponentType a, ComponentType b);
        public abstract ComponentType Subtract(ComponentType a, ComponentType b);
        public abstract ComponentType Multiply(ComponentType a, ComponentType b);
        public abstract ComponentType Divide(ComponentType a, ComponentType b);
        public abstract ComponentType Negative(ComponentType a);
        public abstract ComponentType Inverse(ComponentType a);
        public abstract ComponentType Conjugate(ComponentType a);

        /// <summary>Matrix addition.</summary>
        public void Add(Matrix<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Matrix addition: first operand is null.");
            if (a.d1 <= 0 || a.d1 <= 0)
                throw new ArgumentException("Matrix addition: dimension of operand is not greater thatn 0.");
            if (this.d1 != a.d1 || this.d2!=a.d2)
                throw new ArgumentException("Matrix addition: dimensions of operands are not consistent.");
            for (int i = 0; i < a.d1; ++i)
                for (int j = 0; i < a.d2; ++j)
            {
                this[i,j] = Add(this[i,j], a[i,j]);
            }
        }

        /// <summary>Matrix subtraction.</summary>
        public void Subtract(Matrix<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Matrix subtraction: first operand is null.");
            if (a.d1 <= 0 || a.d1 <= 0)
                throw new ArgumentException("Matrix subtraction: dimension of operand is not greater thatn 0.");
            if (this.d1 != a.d1 || this.d2!=a.d2)
                throw new ArgumentException("Matrix subtraction: dimensions of operands are not consistent.");
            for (int i = 0; i < a.d1; ++i)
                for (int j = 0; i < a.d2; ++j)
            {
                this[i,j] = Subtract(this[i,j], a[i,j]);
            }
        }


        /// <summary>Multiplication of a matrix by a scalar.</summary>
        public void Multiply(ComponentType scalar)
        {
            if (this.d1 < 1)
                throw new ArgumentException("Multiplication of a matrix by a scalar: first matrix dimension is less than 1.");
            if (this.d2 < 1)
                throw new ArgumentException("Multiplication of a matrix by a scalar: second matrix dimension is less than 1.");
            for (int i = 0; i < this.d1; ++i)
            for (int j = 0; j < this.d2; ++j)
            {
                this[i,j] = Multiply(scalar, this[i,j]);
            }
        }


        #endregion // Operations

    }  // class Matrix<>



}  // mnamespace NumLib

