﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace CSharp
{
    
    
    /// <summary>This class demonstrates how to work with files.</summary>
    public class ExampleFiles
    {

        /// <summary>Reads contents of the specified file, copies them to a byte array and returns the array.</summary>
        /// <param name="fileName">Name of the file whose contents is returned in a byte array.</param>
        /// <returns>Byte array containing contents of the file.</returns>
        public static byte[] GetFileContents(string fileName)
        {
            if (!File.Exists(fileName))
                throw new ArgumentException("File does not exist: " + fileName);
            byte[] ret = null;
            FileStream fs = new FileStream(fileName,FileMode.Open,FileAccess.Read);
            try
            {
                if (fs != null)
                {
                    long fileLength = fs.Length;
                    ret = new byte[fileLength];
                    fs.Read(ret, 0 /* offset */, (int) fileLength);
                }
            }
            catch { }
            finally 
            {
                // Close the file stream in the finally block such that this is performed for sure:
                fs.Close();
            }
            return ret;
        }

        /// <summary>Returns contents of the specified file as string in the specified encoding.</summary>
        /// <param name="fileName">Name of the file.</param>
        /// <param name="encoding">Encoding used.</param>
        /// <returns>String containing complete contents of the file.</returns>
        public static string FileToString(string fileName, System.Text.Encoding encoding)
        {
            byte[] contents = GetFileContents(fileName);
            return encoding.GetString(GetFileContents(fileName));
        }



        /// <summary>Returns contents of the specified file as string in the specified encoding.
        /// Unicode encoding (UTF16) is used for string representation.</summary>
        /// <param name="fileName">Name of the file.</param>
        /// <returns>String containing complete contents of the file.</returns>
        public static string FileToString(string fileName)
        {
            return FileToString(fileName, System.Text.Encoding.ASCII);
        }



        /// <summary>Logs a message to a text file.
        /// Date is printed before the message. Messages are appended to existing file contents if the file already exists.</summary>
        /// <param name="fileName">Name of the file to which message is read.</param>
        /// <param name="Message">Message that is logged to the file.</param>
        public static void LogToFile(string fileName, string Message)
        {
            // using block means that writer.Dispose() will be called at the end of the block, even if exception is thrown:
            using (StreamWriter writer = new StreamWriter(fileName, true /* append */, Encoding.Unicode))
            {
                DateTime currentTime = DateTime.Now;
                writer.WriteLine(currentTime.ToString() + ": ");
                writer.WriteLine(Message);
            }
        }

        /// <summary>Appends the specified text to the specified file.</summary>
        /// <param name="fileName">File to which text is appended.</param>
        /// <param name="Message">Text to be appended.</param>
        public static void AppendToFile(string fileName, string text)
        {
            // using block means that writer.Dispose() will be called at the end of the block, even if exception is thrown:
            using (StreamWriter writer = new StreamWriter(fileName, true /* append */, Encoding.Unicode))
            {
                writer.WriteLine(text);
            }
        }


        /// <summary>Writes some test binary data in the specified format to the specified file.
        /// File is rewritten if it already exists.
        /// A string, an integer number, a string, and a double number are written.</summary>
        /// <param name="fileName">Name of the file into which data is written.</param>
        /// <param name="intNum">Integer number that is written to the file.</param>
        /// <param name="doubleNum"></param>
        public static void WriteTestData(string fileName, int intNum, double doubleNum)
        {
            string str1 = "Integer value: ";
            string str2 = "Double value: ";
            using (FileStream fs = new FileStream(fileName, FileMode.Create, FileAccess.Write))
            {
                using (BinaryWriter writer = new BinaryWriter(fs))
                {
                    writer.Write(str1);
                    writer.Write(intNum);
                    writer.Write(str2);
                    writer.Write(doubleNum);
                }
            }
        }

        /// <summary>Reads some test binary data in the specified format from the specified file.
        /// Format must be the same as written by the WriteTestData() method.
        /// A string, an integer number, a string, and a double number are read.
        /// What is read is written to the console.</summary>
        /// <param name="fileName">Name of the file from which data is read.</param>
        public static void ReadTestData(string fileName)
        {
            string str1;
            string str2;
            int intNum;
            double doubleNum;
            using (FileStream fs = new FileStream(fileName, FileMode.Open, FileAccess.Read))
            {
                using (BinaryReader reader = new BinaryReader(fs))
                {
                    str1 = reader.ReadString();
                    intNum = reader.ReadInt32();
                    str2 = reader.ReadString();
                    doubleNum = reader.ReadDouble();
                    Console.WriteLine();
                    Console.WriteLine("The following data has been read from the file " + fileName + ": ");
                    Console.WriteLine(str1);
                    Console.WriteLine(intNum.ToString());
                    Console.WriteLine(str2);
                    Console.WriteLine(doubleNum.ToString());
                    Console.WriteLine();
                    Console.WriteLine();
                }
            }
        }



        public static void Example()
        {
            string textFileName = "TextFile.txt";
            string logFileName = "LogFile.txt";
            string binaryFileName = "BinaryFile.txt";
            string exampleDirectoryRelative = "files";
            
            Console.WriteLine();
            Console.WriteLine("File examples: ");
            Console.WriteLine("Current directory: " + Directory.GetCurrentDirectory());
            Console.WriteLine();
            string exampleDirectory = Path.Combine(
                Path.GetDirectoryName(Path.GetDirectoryName(Directory.GetCurrentDirectory())),
                exampleDirectoryRelative);
            Console.WriteLine("Directory containing example files: " + exampleDirectory);
            if (!Directory.Exists(exampleDirectory))
                throw new IOException("Dierctory containing examples does not exist: " + exampleDirectory);
            string textFile = Path.Combine(exampleDirectory, textFileName);
            string logFile = Path.Combine(exampleDirectory, logFileName);
            string binaryFile = Path.Combine(exampleDirectory, binaryFileName);
            try
            {
                if (!File.Exists(textFile))
                    throw new Exception("Text file does not exist: " + textFile);
                AppendToFile(logFile, Environment.NewLine + Environment.NewLine);
                LogToFile(logFile, "Beginning of demonstration.");
                
                LogToFile(logFile, "Before reading contents of the text file.");
                Console.WriteLine(Environment.NewLine + "Contents of the text file:");
                Console.Write(FileToString(textFile));
                Console.WriteLine();
                LogToFile(logFile, "Text file contents written: " + textFile);
                
                LogToFile(logFile, "Before writing to a binary file...");
                if (!File.Exists(binaryFile))
                    throw new Exception("Binary file does not exist: " + binaryFile);
                WriteTestData(binaryFile, 123, 654.321);
                LogToFile(logFile, "Data written to the binary file: " + binaryFile);
                
                LogToFile(logFile, "Before reading back from the binary file...");
                ReadTestData(binaryFile);
                LogToFile(logFile, "Binary data read from the file: " + binaryFile);
            }
            catch (Exception ex)
            {
                LogToFile(logFile, "Exception occurred: " +  ex.Message);
                throw;
            }
        }



    }



}
