﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


// CLASSES - INTRODUCTION


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace CSharp
{

    /// <summary>Auxiliary structure, stores an int key-value pair.</summary>
    public struct DoubleInt : IComparable<DoubleInt>
    {

        public DoubleInt(int key, int value)
        {
            this.Key = key;
            this.Value = value;
        }

        public int
            Key, 
            Value;

        public override string ToString()
        {
            return Key + ": " + Value;
        }

        /// <summary>Compares the current structure with another.
        /// This method is of IComparable(DoubleInt), which is implemented to enable sorting lists of DoubleInt.</summary>
        /// <param name="other">Structure to which the current structure is compared.</param>
        /// <returns></returns>
        public int CompareTo(DoubleInt other)
        {
            if (this.Value < other.Value)
                return -1;
            else if (this.Value > other.Value)
                return 1;
            else
                return 0;
        }

    }

    /// <summary>Storage for integer key-value pairs with thread safe Add operation.
    /// Adding elements is thread safe.</summary>
    public class DoubleIntStore
    {
        public bool Output = false;

        protected List<DoubleInt> l = new List<DoubleInt>();

        protected object lockObject = new object();

        /// <summary>Adds anew key to the store.</summary>
        public void Add(int key, int value)
        {
            lock (lockObject)
            {
                l.Add(new DoubleInt(key, value));
                if (Output)
                    Console.WriteLine("Added to store from thread " + key + ": ID = " + value);
            }
        }

        /// <summary>Removes all elements from the store.</summary>
        public void Clear()
        {
            lock (lockObject)
            {
                l.Clear();
            }
        }

        public void Sort()
        {
            lock (lockObject)
            {
                l.Sort();
            }
        }


        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            lock (lockObject)
            {
                foreach (DoubleInt pair in l)
                {
                    sb.AppendLine(pair.ToString());
                }
            }
            return sb.ToString();
        }

    }



    /// <summary>A thread safe variant of IdGenerator.</summary>
    /// <remarks>In order to achieve better demonstration of thread unsafety, additional
    /// CPU consuming task implemented by WasteCpu() is performed in the middle of ID generaton.</remarks>
    public class IdGeneratorUnsafe : IdGenerator
    {


        /// <summary>Generates and returns the next unique ID.\
        /// The same as base class' method, except that some CPU consuming task is performed in the middle of
        /// generation in order to emphasise effects of thread unsafety.</summary>
        /// <returns></returns>
        public override int GetId()
        {
            int ret = _nextId;
            WasteCpu(20);
            _nextId += _increment;
            return ret;
        }

        /// <summary>Performs some CPU consuming operation.</summary>
        /// <param name="numIterations">Number of iterations.</param>
        private void WasteCpu(int numIterations)
        {
            double a = 0.5, b = 0.7;
            for (int i = 0; i < numIterations; ++i)
            {
                a = 1.2 + Math.Sin(a);
                b = Math.Cos(1.5 + b);
                a = Math.Exp(0.5*(a + b));
            }
        }

    }

    /// <summary>A thread safe variant of IdGenerator. Blocks where internal 
    /// data is accessed are locked.</summary>
    /// <remarks>In order to implement thread safety properly, we need to know all public methods 
    /// where internal data is accessed. It is a bnetter practice to implement thread safety in 
    /// base classes, but when we get base classes from a third party libraries, a good soltion 
    /// is to derive a thread safe class.</remarks>
    public class IdGeneratorThreadSafe : IdGeneratorUnsafe
    {
        protected object lockObject = new object();

        /// <summary>Generates and returns a new unique (in the scope of the current object) ID.
        /// Thread safe version of the method, calls the base class' method in the lock block.</summary>
        /// <remarks>If the method was not marked virtual then we could not override it. In 
        /// such a case we could only wrap the original class into a proxy class.</remarks>
        public override int GetId()
        {
            // The lock block ensures that only one thread can execute the whole block at a time.
            // Execution of such block is serialized among all threads. 
            // At the beginning of the block, a thread executing it acquires the lock. If the lock
            // is hold by another thread at this moment, the current thread blocks until the lock is
            // released, and then acquires the lock. At the end of the block the thread releases the 
            // lock such that some other thread can acquire it. If several threads requre the lock
            // almost simultanely then the runtime environment ensuers that only one at a time acquires
            // the lock.
            // When the thread blocks because it is waiting for the lock to be released, it is in idle
            // state such that CPU time slices are delivered to another threads (the current thread is
            // skipped by the CPU scheduler while blocked).
            lock (lockObject)
            {
                return base.GetId();
            }
        }

        /// <summary>Returns the next ID that will be generated by the current object.
        /// This is different from GetId() because it does not affect the ID that will be
        /// generated next, it only returns it.
        /// The method is thread safe.</summary>
        public override int WhichIsNext()
        {
            lock (lockObject)
            {
                return base.WhichIsNext();
            }
        }
    }  // IdGeneratorThreadSafe


    /// <summary>Demonstration of thread safe and thread unsafe ID generation.</summary>
    public class ThreadTest
    {

        private static DoubleIntStore idStore;

        private static IdGeneratorUnsafe idGen;

        private static IdGeneratorThreadSafe idGenThreadSafe;

        
        /// <summary>Generates the specified number of unique IDs by the specified ID generator, 
        /// and stores those IDs to the specified ID store.</summary>
        /// <param name="num">Number of generated and stored IDs.</param>
        /// <param name="generator">Generator used to generate IDs.</param>
        /// <param name="store">Storage object where the generated IDs are stored. IDs are stored in form of
        /// DoubleInt structs where the first integer represents the ID of the thread that performed the
        /// oepration, and the second integere represents the generated ID.</param>
        private static void GenerateIds(int num, IdGenerator generator, DoubleIntStore store)
        {
            int threadId = Thread.CurrentThread.ManagedThreadId;
            for (int i = 0; i < num; ++i)
            {
                store.Add(threadId, generator.GetId());
            }
        }

        private static int numGenerationsPerThread = 50;

        /// <summary>Method to be executed in parallel by two threads. 
        /// The method is thread unsafe because it uses the thread unsafe ID generator.</summary>
        private static void ThreadMethodUnsafe()
        {
            GenerateIds(numGenerationsPerThread, idGen, idStore);
        }

        /// <summary>Method to be executed in parallel by two threads. 
        /// The method is thread safe because it uses the thread safe ID generator.</summary>
        private static void ThreadMethodSafe()
        {
            GenerateIds(numGenerationsPerThread, idGenThreadSafe, idStore);
        }


        /// <summary>Unique IDs are simultaneously generated in two parallel threads.
        /// In the first case the thread unsafe ID generator is used, and in the second case the thread
        /// safe generator is used.
        /// The generated IDs are stored such that they can be printed out when both thread complete the work.
        /// The </summary>
        /// <param name="numGenerations">Number of generated IDs per thread; recommended value is 50.</param>
        public static void ExampleParallelIdGeneration(int numGenerations)
        {
            Thread thread1, thread2;
            numGenerationsPerThread = numGenerations;
            // Allocate storage and ID generators:
            idStore = new DoubleIntStore();
            idGen = new IdGeneratorUnsafe();
            idGenThreadSafe = new IdGeneratorThreadSafe();
            // Generation of IDs by a thread unsafe generator:
            Console.WriteLine();
            Console.WriteLine("Starting threads, UNSAFE ID generation, " + numGenerations 
                + " IDs per thread...");
            // Create parallel threads; parameters to thread constructors are threads that will be called by the 
            // threads (the thread only calls the method and stops).
            thread1 = new Thread(ThreadMethodUnsafe);
            thread2 = new Thread(ThreadMethodUnsafe);
            // Clear the storage:
            idStore.Clear();
            // Start both threads:
            thread1.Start();
            thread2.Start();
            // The current thread waits until both threads stop:
            thread1.Join();
            thread2.Join();
            // Output the generated IDs that were stored by both threads to  idStore:
            Console.WriteLine("Generated UNSAFE IDs (the first number denotes the thread):");
            Console.WriteLine(idStore.ToString());
            Console.WriteLine();

            // Generation of IDs by a thread safe generator:
            Console.WriteLine();
            Console.WriteLine("Starting threads, THREAD SAFE ID generation, " + numGenerations 
                + " IDs per thread...");
            // Create parallel threads; parameters to thread constructors are threads that will be called by the 
            // threads (the thread only calls the method and stops).
            thread1 = new Thread(ThreadMethodSafe);
            thread2 = new Thread(ThreadMethodSafe);
            // Clear the storage:
            idStore.Clear();
            // Start both threads:
            thread1.Start();
            thread2.Start();
            // The current thread waits until both threads stop:
            thread1.Join();
            thread2.Join();
            // Output the generated IDs that were stored by both threads to  idStore:
            Console.WriteLine("Generated SAFE IDs (the first number denotes the thread):");
            Console.WriteLine(idStore.ToString());
            Console.WriteLine();
        }


    }  // class ThreadTest




}
