﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


// CLASSES - INTRODUCTION


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CSharp
{


    /// <summary>Generates unique IDs.</summary>
    public class IdGeneratorBase 
    {

        /// <summary>Constructor.</summary>
        /// <param name="firstId">The first ID that is generated by this object.</param>
        /// <param name="increment">Integer by which subsequently generated IDs are incremented.</param>
        public IdGeneratorBase(int firstId, int increment)
        {
            if (increment == 0)
                throw new ArgumentException("ID increment can not be 0.");
            this._nextId = firstId;
            this._increment = increment;
        }

        /// <summary>Default constructor.</summary>
        public IdGeneratorBase() : this(1,1)
        {  }

        protected int _nextId = 1;
        protected int _increment = 1;

        /// <summary>Generates and returns a new unique (in the scope of the current object) ID.</summary>
        /// <returns>A new unique ID.</returns>
        /// <remarks>virtual means that the method can be overridden in derived classes.</remarks>
        public virtual int GetId()
        {
            int ret = _nextId;
            _nextId += _increment;
            return ret;
        }

        public override string ToString()
        { return _nextId.ToString(); }

    }


    /// <summary>Generates unique IDs.</summary>
    public class IdGenerator: IdGeneratorBase
    {

        /// <summary></summary>
        public IdGenerator() : base(1,1)
        {  }

        /// <summary>Returns the next ID that will be generated by the current object.
        /// This is different from GetId() because it does not affect the ID that will be
        /// generated next, it only returns it.</summary>
        /// <remarks>virtual means that tha method can be overridden in subclasses.</remarks>
        public virtual int WhichIsNext()
        {
            return _nextId;
        }

    }

    /// <summary>Generates even unique IDs.</summary>
    public class IdGeneratorEven : IdGenerator
    {

        // Default constructor, ensures that the first generated ID equals 2
        public IdGeneratorEven()
        {
            _nextId = 2;
            _increment = 2;
        }

        public IdGeneratorEven(int firstId)
        {
            if (firstId % 2 != 0)
                throw new ArgumentException("Argument should be an even number.");
            _nextId = firstId;
            _increment = 2;
        }

    }


    /// <summary>Interface for classes whose objects have unique IDs.</summary>
    public interface IIdentifiable 
    {
        /// <summary>Gets ID of the current object.</summary>
        int Id 
        {
            get;
        }
    }

    /// <summary>Interface for classes that define a person.</summary>
    public interface IPerson: IIdentifiable
    {
        string Name
        { get; }

        string SecondName
        {
            get;
        }

        /// <summary>Age in years.</summary>
        int Age
        {
            get;
            set;
        }

        /// <summary>Car owned by this person.</summary>
        Car Car
        {
            get;
            set;
        }
    }



    /// <summary>Person representation with its data.</summary>
    public class Person: IPerson
    {

        protected static IdGeneratorBase _idGenerator  = new IdGenerator();

        /// <summary>Private default constructor prevents initialization without arguments.</summary>
        private Person()
        {
            Id = _idGenerator.GetId();
        }


        public Person(string name, string secondName) : this()
        {
            if (string.IsNullOrEmpty(name))
                throw new ArgumentNullException("Person's name not specified (null argument).");
            if (string.IsNullOrEmpty(secondName))
                throw new ArgumentNullException("Person''s second name not specified (null argument)");
            this._name = name;
            this._secondName = secondName;
            this._age = -1;
        }

        public Person(string name, string secondName, int age) : this(name, secondName)
        {
            this._age = age;
        }

        private string _name, _secondName;

        private int _age = 0, _id = 0;

        private Car _car;

        public int Id
        {
            get { return this._id; }
            protected set { this._id = value; }
        }

        /// <summary>Gets person's name.</summary>
        public string Name
        { 
            get { return _name; }
        }

        /// <summary>Gets person's second name.</summary>
        public string SecondName
        {
            get { return _secondName; }
        }

        /// <summary>Gets or sets p erson's age.</summary>
        public int Age
        {
            get
            {
                return this._age;
            }
            set
            {
                this._age = value;
            }
        }

        /// <summary>Gets or sets the car owned by the particular person.</summary>
        public Car Car
        {
            get { return _car; }
            set { _car = value; }
        }

        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine(Name + " " + SecondName + ", ID = " + Id);
            if (Age > 0)
                sb.AppendLine("  Age: " + Age);
            else
                sb.AppendLine("  Age unknown.");
            if (Car==null)
                sb.AppendLine("  Does not have a car.");
            else 
                sb.AppendLine("  Drives " + Car);
            return sb.ToString();
        }

    } // class Person





    public class Car : IIdentifiable
    {

        protected static IdGeneratorBase _idGenerator = new IdGenerator();

        /// <summary>Prevents calling argument-less constructor (since car must have brand 
        /// and production year), but initializes ID, therefore it should be called in other constructors.</summary>
        private Car()
        {
            Id = _idGenerator.GetId();
        }

        /// <summary>Constructor. Calls argumentless constructor in order to </summary>
        /// <param name="brand">Brand name.</param>
        /// <param name="year">Year of production.</param>
        public Car(string brand, int year) : this()
        {
            this._brand = brand;
            this._year = year;
        }

        private string _brand;

        private int _year, _id;

        /// <summary>ID of the car.</summary>
        public int Id
        {
            get { return this._id; }
            protected set { this._id = value; }
        }


        /// <summary>Car's brand.</summary>
        public string Brand
        {
            get { return _brand; }
        }

        /// <summary>Year when car was produced.</summary>
        public int Year
        { get { return _year; } }

        public override string  ToString()
        {
            return Brand + " " + Year + ", ID = " + Id ;  
        }

    }


    /// <summary>Car with even ID.</summary>
    public class CarWithEvenId : Car
    {
        
        public CarWithEvenId(string brand, int year)
            : base(brand, year)
        {
            if (Id % 2 != 0)
                Id = _idGenerator.GetId();
        }

    }

    



    /// <summary>Examples for classes.</summary>
    public class ExampleClasses
    {

        /// <summary>Use of the IdGenerator class.</summary>
        public static void IdGenerator()
        {
            IdGenerator gen1 = new IdGenerator();
            IdGenerator gen2 = new IdGeneratorEven();

            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen1: " + gen1.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
            Console.WriteLine("Id generated by gen2: " + gen2.GetId());
        }


        /// <summary>Classes that use IdGenerator class.</summary>
        public static void Classes()
        {
            IPerson p1, p2, p3, p4;

            p1 = new Person("Marija", "Novak", 43);
            p1.Car = new Car("Peugeot", 2005);
            p2 = new Person("Tony", "Crook");
            p2.Car = new Car("BMW", 2008);
            p3 = new Person("Kim", "Lee");
            p3.Car = new CarWithEvenId("Toyota", 2020);
            p4 = new Person("Pjotr", "Sobietzky", 26);

            Console.WriteLine("p1: " + Environment.NewLine + p1);
            Console.WriteLine("p2: " + Environment.NewLine + p2);
            Console.WriteLine("p3: " + Environment.NewLine + p3);
            Console.WriteLine("p4: " + Environment.NewLine + p4);
        }


        /// <summary>Demonstrates upcasting, downcasting and operators int. </summary>
        public static void Casting()
        {
            IdGenerator gen1, gen2;
            IdGeneratorEven genEven;

            gen1 = new IdGenerator();
            Console.WriteLine("gen1 assigned - new IdGenerator");

            gen2 = new IdGeneratorEven();
            Console.WriteLine("Upcasting: gen1 assigned - new IdGeneratorEven");

            genEven = (IdGeneratorEven) gen2;
            Console.WriteLine("Downcasting: gen2 assigned to genEven");

            //// Wrong downcasting: this would generate an error: 
            //genEven = (IDGeneratorEven) gen1;
            //Console.WriteLine("Downcasting: gen1 assigned to genEven");

            // Operator is:
            genEven = new IdGeneratorEven();
            Console.WriteLine();
            Console.WriteLine("Checking with operator is:" + Environment.NewLine);
            Console.WriteLine("Variable declared as derived type:");
            if (genEven is IdGenerator)
                Console.WriteLine("genEven IS IdGenerator.");
            else
                Console.WriteLine("genEven is NOT IdGenerator.");
            if (genEven is IdGeneratorEven)
                Console.WriteLine("genEven IS IdGeneratorEven.");
            else
                Console.WriteLine("genEven is NOT IdGeneratorEven.");

            Console.WriteLine();
            Console.WriteLine("Variables defined as base type:");
            if (gen1 is IdGeneratorEven)
            {
                Console.WriteLine("gen1 IS IdGeneratorEven.");
                genEven = (IdGeneratorEven) gen1;
            } else
            {
                Console.WriteLine("gen1 is NOT IdGeneratorEven.");
                genEven = null;
            }
            if (gen2 is IdGeneratorEven)
            {
                Console.WriteLine("gen2 IS IdGeneratorEven.");
                genEven = (IdGeneratorEven) gen2;
            } else
            {
                Console.WriteLine("gen2 is NOT IdGeneratorEven.");
                genEven = null;
            }

            Console.WriteLine();
            Console.WriteLine("Safe downcasting with as operator: ");
            Console.WriteLine("Assigning gen1 to genEven: ");
            genEven = gen1 as IdGeneratorEven;
            Console.WriteLine("Downcasting performed, genEven = " + (genEven == null? "null": genEven.ToString()));

            Console.WriteLine("Assigning gen2 to genEven: ");
            genEven = gen2 as IdGeneratorEven;
            Console.WriteLine("Downcasting performed, genEven = " + (genEven == null ? "null" : genEven.ToString()));

        }

    }  // class ExampleClasses



}
