﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NumLib
{


    partial class Differential
    {

        /// <summary>Implementation of Runge-Kutta method for a single first order initial value problem,
        /// error is O(h^5).</summary>
        /// <param name="f">Function that relates derivative of the function with independent variable and function value.</param>
        /// <param name="from"></param>
        /// <param name="to"></param>
        /// <param name="numint"></param>
        /// <param name="y0"></param>
        /// <returns></returns>
        public static double[] RungeKutta(ScalarFunction f, double from, double to, int numint, double y0)
        {

            double h = (to - from) / numint;
            double[] ret = new double[numint + 1];
            ret[0] = y0;
            double[] arg = new double[2];
            for (int i = 0; i < numint; ++i)
            {
                double x = from + i * h;
                double y = ret[i];
                arg[0] = x;
                arg[1] = y;
                double k1 = h * f(arg);
                arg[0] = x + h / 2;
                arg[1] = y + k1 / 2;
                double k2 = h * f(arg);
                arg[1] = y + k2 / 2;
                double k3 = h * f(arg);
                arg[1] = y + k3;
                double k4 = f(arg);
                ret[i + 1] = ret[i] + (k1 + 2 * k2 + 2 * k3 + k4) / 6;
            }
            return ret;
        }



        /// <summary>Implementation of Runge-Kutta method for a a system first order initial value problems,
        /// error is O(h^5).</summary>
        /// <param name="f">Array of functions of independent and dependent variables whose values equal solution derivatives.</param>
        /// <param name="y0">Initial condition / values of functions in the initial point (from).</param>
        /// <param name="from">Lower limit of the interval where solutions will be be calculated.</param>
        /// <param name="to">Upper limit of the interval where solutions will be be calculated.</param>
        /// <param name="numint">Number of sub-intervals on which integration interal is divided.</param>
        /// <param name="x">Output parameter. Table of x values (nodes) at which function values are calculated.</param>
        /// <param name="y">Output parameter. Table of tables of values approximate solutions.</param>
        public static void RungeKuttaSystem(ScalarFunction[] f, double[] y0, double from, double to, int numint,
            out double[] x, out double[,] y)
        {
            if (f == null)
                throw new ArgumentNullException("Table of functions is not specified (null reference).");
            if (y0 == null)
                throw new ArgumentNullException("Table of initial values is not specified (null reference).");
            if (numint < 1)
                throw new ArgumentException("Number of sub-intervals should be at least 1, specified: " + numint.ToString());
            if (f.Length < 1)
                throw new ArgumentException("Number of functions should be at least 1, specified: "+ f.Length.ToString());
            if (f.Length != y0.Length)
                throw new ArgumentException("Number of functions (" + f.Length.ToString() + ") does not mach number of initial conditions ("
                    + y0.Length.ToString() + ").");
            double h = (to - from) / (double) numint;
            int numpoints = numint + 1;
            int numequations = f.Length;
            x = new double[numpoints];
            y = new double[numequations, numpoints];
            // Fill table of x-values:
            for (int i = 0; i < numpoints; ++i)
                x[i] = from + i * h;
            // Copy initial values in the table of y-values (functions that we):
            for (int j = 0; j < numequations; ++j)
                y[j, 0] = y0[j];
            
            // Start main solution loop, initialize auxiliary storage first:
            double[] funcarg = new double[numequations + 1];  // used as function argument for f[j]
            double[]
                k1 = new double[numequations],
                k2 = new double[numequations],
                k3 = new double[numequations],
                k4 = new double[numequations];
            for (int i = 0; i < numint; ++i)
            {
                // Iterate over all nodes (except the last one):
                // Calculate vector of coefficients k1 (prepare arguments of f[j] first):
                funcarg[0] = x[i];
                for (int j = 0; j < numequations; ++j)
                    funcarg[j + 1] = y[j,i];
                for (int j = 0; j < numequations; ++j)
                    k1[j] = h * f[j](funcarg);
                // Calculate vector of coefficients k2 (prepare arguments of f[j] first):
                funcarg[0] = x[i] + h / 2;
                for (int j = 0; j < numequations; ++j)
                    funcarg[j + 1] = y[j,i] + k1[j] / 2;
                for (int j = 0; j < numequations; ++j)
                    k2[j] = h * f[j](funcarg);
                // Calculate vector of coefficients k3 (prepare arguments of f[j] first):
                funcarg[0] = x[i] + h / 2;
                for (int j = 0; j < numequations; ++j)
                    funcarg[j + 1] = y[j,i] + k2[j] / 2;
                for (int j = 0; j < numequations; ++j)
                    k3[j] = h * f[j](funcarg);
                // Calculate vector of coefficients k4 (prepare arguments of f[j] first):
                funcarg[0] = x[i] + h;
                for (int j = 0; j < numequations; ++j)
                    funcarg[j + 1] = y[j,i] + k3[j];
                for (int j = 0; j < numequations; ++j)
                    k4[j] = h * f[j](funcarg);
                // Calculate approximation of unknown functions in the next point:
                for (int j = 0; j < numequations; ++j)
                {
                    y[j, i + 1] = y[j, i] + (k1[j] + 2 * k2[j] + 2 * k3[j] + k4[j]) / 6.0;
                }
            }
        }
    }


    partial class Differential
    {



    }


}
