﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NumLib;

// Za uporabo grafike:
using System.Windows.Forms;
using System.Drawing;
using ZedGraph;
// using ZGControlTest;

namespace ProgramDifferential
{
    class ProgramDifferential
    {

        // DUŠENO NIHANJE:

        // Fizikalni parametri problema:
        static double
            omega0 = 1.0,  // kotna frekvenca nedušenega nihala
            kappa = 0.1,   // linearni koeficient dušenja
            A0 = 1.0,      // amplituda nihala
            v0 = 0.0;      // začetna hitrost

        // funkciji, ki določata diferencialni enačbi za x in v:

        public static double fx(double[] arg)
        {
            if (arg == null)
                throw new ArgumentNullException("Argumenti funkcije za izračun odvoda prve iskane funkcije niso podani.");
            if (arg.Length != 3)
                throw new ArgumentException("Tabela argumentov bi morala imeti 3 elemente, število pri klicu: "
                    + arg.Length + ".");
            // Izvlečemo argumente po pomenu:
            double 
                t = arg[0],
                x = arg[1],
                v = arg[2];
            return v;
        }

        public static double fv(double[] arg)
        {
            if (arg == null)
                throw new ArgumentNullException("Argumenti funkcije za izračun odvoda druge iskane funkcije niso podani.");
            if (arg.Length != 3)
                throw new ArgumentException("Tabela argumentov bi morala imeti 3 elemente, število pri klicu: "
                    + arg.Length + ".");
            // Izvlečemo argumente po pomenu:
            double 
                t = arg[0],
                x = arg[1],
                v = arg[2];
            return -2 * kappa * v - omega0 * omega0 * x; ;
        }


        public static void Main(string[] args)
        {
            // Numerični parametri:
            double
                from = 0.0,         // spodnja meja intervala računanja
                to = 4 * 2 * Math.PI / omega0 ;  // zgornja meja intervala računanja
            int numint = 20 * 4;  // število intervalov

            try
            {
                // Priprava podatkov za metodo Runge-Kutta za sistem dif. enačb 1. reda:
                // Začetni vrednosti:
                double[] y0 = new double[2];
                y0[0] = A0;  // začet. pogoj za x
                y0[1] = v0;  // začet. pogoj za v
                ScalarFunction[] f = new ScalarFunction[2];
                f[0] = fx;
                f[1] = fv;

                // Izračun neznanih funkcij z metodo Runge-Kutta:
                double[] x;   // tabela časov
                double[,] y;  // tabela pomikov in hitrosti
                Differential.RungeKuttaSystem(f, y0, from, to, numint, out x, out y);
                // Izpišemo x(t);
                PrintResults(x, y, false /* PrintVelocities */, omega0);


                // Izris rezultatov:
                // Priprava rezultatov za izris: 
                int n = x.Length;  // število točk
                X1 = new double[n];
                Y1 = new double[n];
                X2 = new double[n];
                Y2 = new double[n];
                for (int i = 0; i < n; ++i)
                {
                    X1[i] = X2[i] = x[i]; // časi
                    Y1[i] = y[0, i];  // pomiki
                    Y2[i] = y[1, i];  // hitrosti
                }
                // Izris:
                // Naredimo novo kontrolo tipa ZedGraphWindow:
                ZedGraphWindow Zwin = new ZedGraphWindow();
                // Na kontroli nastavimo metodo, ki bo izrisala stvari:
                Zwin.PlotDelegate = ParametricPlot;
                // Odpremo kontrolo:
                Application.Run(Zwin);



            }
            catch (Exception ex)
            {
                Console.WriteLine(Environment.NewLine);
                Console.WriteLine("NAPAKA: " +  ex.Message + Environment.NewLine + Environment.NewLine);
            }

        }



        /// <summary>Izpis rezultatov.</summary>
        /// <param name="tabt">Tabela časov.</param>
        /// <param name="taby">Tabela izračunanih pomikov in hitrosti.</param>
        /// <param name="PrintVelocities">Določa, ali naj se poleg pomikov izpisujejo tudi hitrosti.</param>
        /// <param name="omega">Kotna frekvenca. Če je večja od 0, se uporabi za izpis presledkov pri posebnih časih.</param>
        public static void PrintResults(double[] tabt, double[,] taby, bool PrintVelocities, double omega)
        {
            if (tabt == null || taby == null)
                throw new ArgumentNullException("Tabei časov in pomikov nista podani.");
            if (tabt.Length < 2 || taby.GetLength(1) != tabt.Length)
                throw new ArgumentException("Tabeli časov in pomikov nista konsistentni.");
            Console.WriteLine(Environment.NewLine);
            Console.WriteLine("Tabela x(t) (" + tabt.Length.ToString() + " točk): ");
            double nextspacingtime = 1.0e15, spacingFi = 2.0 * Math.PI / 1.0;
            if (omega > 0)
                nextspacingtime = tabt[0] + (spacingFi) / omega;
            for (int i = 0; i < tabt.Length; ++i)
            {
                Console.Write((i + 1).ToString() + ": t = " + tabt[i].ToString() + ", x(t) = " + taby[0, i].ToString());
                if (PrintVelocities)
                    Console.WriteLine(", v(t) = " + taby[0, i].ToString());
                else
                    Console.WriteLine();
                if (omega > 0 && tabt[i] >= nextspacingtime)
                {
                    // Insert additional spacing in specific intervals:
                    Console.WriteLine();
                    nextspacingtime += spacingFi / omega;
                }
            }
            Console.WriteLine(Environment.NewLine);
        }






        // PODATKI, ki jih uporablja metoda za izris:
        // Teh podatkov ne moremo prenesti preko argumentov (ker so tipi argumentov 
        // metode vnaprej določeni), zato morajo biti neposredno dostopni v telesu
        // metode. To dosežemo ali tako, da metodo in podatek definiramo kot static,
        // ali pa tako, da so definirani znotraj istega razreda.
        static double[] X1 = null, X2 = null, Y1 = null, Y2 = null;
        // Nastavitev, ki določa, ali sta skali za krivulji enaki:
        static bool PlotProportional = true;
        // Izris legende in škatle z namigom:
        static bool
            PlotLegend = true,
            PlotTip = false,
            PlotBoth = true;  // if true then 2 graphs are plotted.


        /// <summary>Sets properties and plotted object on the ZedGraphControl.</summary>
        /// <param name="PlotPane">ZedGraphControl object that on which this method plots.</param>
        public static void ParametricPlot(ZedGraphControl GraphControl)
        {

            if (X1 == null)
                throw new Exception("First X table is null.");
            if (Y1 == null)
                throw new Exception("First Y table is null.");
            if (X1.Length != Y1.Length || X1.Length < 1)
                throw new Exception("Table lengths for the first graph are not consistent.");
            if (PlotBoth)
            {
                if (X2 == null)
                    throw new Exception("Second X table is null.");
                if (Y2 == null)
                    throw new Exception("Second Y table is null.");
                if (X2.Length != Y2.Length || X2.Length < 1)
                    throw new Exception("Table lengths for the second graph are not consistent.");
            }

            // Naredimo seznama točk, ki ju bomo izrisali:
            PointPairList PointList1 = new PointPairList();
            PointPairList PointList2 = new PointPairList();

            // Skopiramo naše podatke v seznama:
            for (int i = 0; i < X1.Length; ++i)
            {
                PointList1.Add(X1[i], Y1[i]);
            }
            if (PlotBoth)
            {
                for (int i = 0; i < X2.Length; ++i)
                {
                    PointList2.Add(X2[i], Y2[i]);
                }
            }

            // Iz seznamov točk naredimo krivulji, ki ju dodamo grafu.
            // Če želimo izrisati samo en graf, naredimo samo eno krivuljo:

            // Get a reference to the GraphPane instance in the ZedGraphControl
            GraphPane myPane = GraphControl.GraphPane;

            // Specify descriptions for the legend:
            string LegendString1 = "x(t)";
            string LegendString2 = "v(t(";
            if (!PlotLegend)
            {
                LegendString1 = LegendString2 = null;
            }

            // Generate a red curve with diamond symbols and LegendString1 in the legend
            LineItem myCurve = myPane.AddCurve(LegendString1,
                PointList1, Color.Red, SymbolType.Diamond);
            // Fill the symbols with white
            myCurve.Symbol.Fill = new Fill(Color.White);

            if (PlotBoth)
            {
                // Generate a blue curve with circle symbols, and LegendString2 in the legend
                myCurve = myPane.AddCurve(LegendString2,
                    PointList2, Color.Blue, SymbolType.Circle);
                // Fill the symbols with white
                myCurve.Symbol.Fill = new Fill(Color.White);
                // Associate this curve with the Y2 axis
            }

            if (!PlotProportional)
            {
                // POZOR: S tem določimo, da imamo za drugo krivuljo drugi osi x os in y in je zato
                // drugače skalirana. Ne glede na to, kakšn je razlika v razponu vrednosti , bosta 
                // krivulji vedno zavzeli približno enak del grafa.
                // Brez spodnjega stavka bi bili izrisani v dejanskih proporcah.
                myCurve.IsX2Axis = true;
                if (PlotBoth)
                    myCurve.IsY2Axis = true;
            }
            else
            {
                myCurve.IsX2Axis = false;
                myCurve.IsY2Axis = false;
            }


            // NASTAVITVE ZA IZRIS GRAFA:
            // Od tu naprej so razne nastavitve, ki vplivajo na to, kaj in kako se na grafu
            // izriše:

            // Set the titles and axis labels
            myPane.Title.Text = "Izris parametričnih krivulje (X(t), Y(t))";
            myPane.XAxis.Title.Text = "t";
            myPane.YAxis.Title.Text = "x(t)";
            if (PlotBoth)
            {
                myPane.X2Axis.Title.Text = "t";
                myPane.Y2Axis.Title.Text = "v(t)";
            }

            // Show the x axis grid
            myPane.XAxis.MajorGrid.IsVisible = true;

            // Make the Y axis scale red
            myPane.YAxis.Scale.FontSpec.FontColor = Color.Red;
            myPane.YAxis.Title.FontSpec.FontColor = Color.Red;
            // turn off the opposite tics so the Y tics don't show up on the Y2 axis
            myPane.YAxis.MajorTic.IsOpposite = false;
            myPane.YAxis.MinorTic.IsOpposite = false;
            // Don't display the Y zero line
            myPane.YAxis.MajorGrid.IsZeroLine = false;
            // Align the Y axis labels so they are flush to the axis
            myPane.YAxis.Scale.Align = AlignP.Inside;
            // Manually set the axis range
            myPane.YAxis.Scale.Min = -30;
            myPane.YAxis.Scale.Max = 30;

            if (PlotBoth)
            {
                // Enable the Y2 axis display
                myPane.Y2Axis.IsVisible = true;
                // Make the Y2 axis scale blue
                myPane.Y2Axis.Scale.FontSpec.FontColor = Color.Blue;
                myPane.Y2Axis.Title.FontSpec.FontColor = Color.Blue;
                // turn off the opposite tics so the Y2 tics don't show up on the Y axis
                myPane.Y2Axis.MajorTic.IsOpposite = false;
                myPane.Y2Axis.MinorTic.IsOpposite = false;
                // Display the Y2 axis grid lines
                myPane.Y2Axis.MajorGrid.IsVisible = true;
                // Align the Y2 axis labels so they are flush to the axis
                myPane.Y2Axis.Scale.Align = AlignP.Inside;
            }

            // Fill the axis background with a gradient
            myPane.Chart.Fill = new Fill(Color.White, Color.LightGray, 45.0f);

            if (PlotTip)
            {
                // Add a text box with instructions
                TextObj text = new TextObj(
                    "Zoom: left mouse & drag\nPan: middle mouse & drag\nContext Menu: right mouse",
                    0.05f, 0.95f, CoordType.ChartFraction, AlignH.Left, AlignV.Bottom);
                text.FontSpec.StringAlignment = StringAlignment.Near;
                myPane.GraphObjList.Add(text);
            }

            // Enable scrollbars if needed
            GraphControl.IsShowHScrollBar = true;
            GraphControl.IsShowVScrollBar = true;
            GraphControl.IsAutoScrollRange = true;
            GraphControl.IsScrollY2 = true;

        }  //





    }
}
