﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NumLib
{
    /// <summary>Numerical calculation of definite integrals.</summary>
    public class Integration  
    {

        // Integration functions that take the function (delegate) as an argument:

        // Variants of RECTANGULAR rule:

        /// <summary>Calculates a numerical integral of a real-valued function by using the rectengular rule.</summary>
        /// <param name="f">The function that is integrated.</param>
        /// <param name="a">Lower limit of the integral.</param>
        /// <param name="b">Upper limit of the integral.</param>
        /// <param name="n">Number of equidistand intervals used for integration. The function is evaluated in
        /// all midpoints of these intervals (n evaluations).</param>
        /// <returns></returns>
        public static double IntegralRectangle(RealFunction f, double a, double b, int n)
        {
            if (n<1)
                throw new ArgumentException("Number of dubintervals should be at least 1. Currently: " 
                    + n.ToString() + ".");
            if (f == null)
                throw new ArgumentNullException("The function to be integrated is not specified (null reference).");
            double h = (b - a) / (double)n;
            // Evaluate function in teh lower interval limit:
            double 
                xl = a,  // left-hand endpoint of the subinterval
                xr = 0,  // right-hand endpoint
                fc,  // function value in the middle of the current subinterval
                integral = 0;  // accummulated value of the integral
            // Iterate over all intervals:
            for (int i = 0; i < n; ++i)
            {
                xr = xl + h;
                fc = f(0.5*(xl+xr));  // Calculate function value in the middle of the interval
                integral += h * fc;
                xl = xr;
            }
            // Check that the right endpoint of the last interval is close enough to the right integral limit:
            double errorsize, stepsize;
            errorsize = Math.Abs(b - xr);
            stepsize = Math.Abs(h);
            if (errorsize > stepsize)
                throw new Exception("Error in interval accummulation (" + errorsize
                    + ") is greater than step size (" + stepsize + ").");
            return integral;
        }

        /// <summary>Calculates a numerical integral of a real-valued function by using the rectengular rule.
        /// The calculated value is an UPPER bound for the integral in the case of monotonous functions.</summary>
        /// <param name="f">The function that is integrated.</param>
        /// <param name="a">Lower limit of the integral.</param>
        /// <param name="b">Upper limit of the integral.</param>
        /// <param name="n">Number of equidistand intervals used for integration. The function is evaluated in
        /// all midpoints of these intervals (n evaluations).</param>
        /// <returns></returns>
        public static double IntegralRectangleHigh(RealFunction f, double a, double b, int n)
        {
            if (n < 1)
                throw new ArgumentException("Number of dubintervals should be at least 1. Currently: "
                    + n.ToString() + ".");
            if (f == null)
                throw new ArgumentNullException("The function to be integrated is not specified (null reference).");
            double h = (b - a) / (double)n;
            // Evaluate function in teh lower interval limit:
            double
                xl = a,  // left-hand endpoint of the subinterval
                xr = 0,  // right-hand endpoint
                fl = f(a),   // function value at xl
                fr=0,   // function value at xr
                fmax = 0, // the greater value among fl and fr
                integral = 0;  // accummulated value of the integral
            // Iterate over all intervals:
            for (int i = 0; i < n; ++i)
            {
                xr = xl + h;
                fr = f(xr);  // calculate the function at the right endpoint of the current subinterval
                             // (we already have the left value)
                // Take the greater of the two function values:
                if (fl>fr)
                    fmax=fl;
                else
                    fmax=fr;
                integral += h * fmax;
                xl = xr;
                fl = fr;
            }
            // Check that the right endpoint of the last interval is close enough to the right integral limit:
            double errorsize, stepsize;
            errorsize = Math.Abs(b - xr);
            stepsize = Math.Abs(h);
            if (errorsize > stepsize)
                throw new Exception("Error in interval accummulation (" + errorsize
                    + ") is greater than step size (" + stepsize + ").");
            return integral;
        }

        /// <summary>Calculates a numerical integral of a real-valued function by using the rectengular rule.
        /// The calculated value is an LOWER bound for the integral in the case of monotonous functions.</summary>
        /// <param name="f">The function that is integrated.</param>
        /// <param name="a">Lower limit of the integral.</param>
        /// <param name="b">Upper limit of the integral.</param>
        /// <param name="n">Number of equidistand intervals used for integration. The function is evaluated in
        /// all midpoints of these intervals (n evaluations).</param>
        /// <returns></returns>
        public static double IntegralRectangleLow(RealFunction f, double a, double b, int n)
        {
            if (n < 1)
                throw new ArgumentException("Number of dubintervals should be at least 1. Currently: "
                    + n.ToString() + ".");
            if (f == null)
                throw new ArgumentNullException("The function to be integrated is not specified (null reference).");
            double h = (b - a) / (double)n;
            // Evaluate function in teh lower interval limit:
            double
                xl = a,  // left-hand endpoint of the subinterval
                xr = 0,  // right-hand endpoint
                fl = f(a),   // function value at xl
                fr = 0,   // function value at xr
                fmin = 0, // the greater value among fl and fr
                integral = 0;  // accummulated value of the integral
            // Iterate over all intervals:
            for (int i = 0; i < n; ++i)
            {
                xr = xl + h;
                fr = f(xr);  // calculate the function at the right endpoint of the current subinterval
                // (we already have the left value)
                // Take the greater of the two function values:
                if (fl < fr)
                    fmin = fl;
                else
                    fmin = fr;
                integral += h * fmin;
                xl = xr;
                fl = fr;
            }
            // Check that the right endpoint of the last interval is close enough to the right integral limit:
            double errorsize, stepsize;
            errorsize = Math.Abs(b - xr);
            stepsize = Math.Abs(h);
            if (errorsize > stepsize)
                throw new Exception("Error in interval accummulation (" + errorsize
                    + ") is greater than step size (" + stepsize + ").");
            return integral;
        }


        // TRAPEZOIDAL RULE:
        // This method samples function values and calls another function that operates on the sampled
        // data (i.e. calculated function values in a discrete set of points).
        // Note that these two functions have the same names but different arguments. This is called
        // "function overloading". 

        public static double IntegralTrapezoidal(RealFunction f, double a, double b, int n)
        {
            if (n < 1)
                throw new ArgumentException("Number of dubintervals should be at least 1. Currently: "
                    + n.ToString() + ".");
            if (f == null)
                throw new ArgumentNullException("The function to be integrated is not specified (null reference).");
            // Allocate space for lists of sampling points and function values in these points:
            double[]
                xList = new double[n+1],
                yList = new double[n+1];
            double h = (b - a) / (double) n;
            double
                x = a;
            // Calculate sempling points and function values in these points:
            for (int i = 0; i <= n; ++i)
            {
                x = a + h *  (double) i;
                xList[i] = x;
                yList[i] = f(x);
            }
            // Calculate the integral according to the trapezoidal rule using the sampled data:
            return IntegralTrapezoidal(xList, yList);
        }


        // Integratning functions that take a list of independent variables and a list of function 
        // values as arguments:
        // Advantage of such functions is that they can save function evaluations because different rules
        // can be applied to the same data sets that are calculated only once. On the other hand, this means
        // higher requests for storage space.
        // The advantage of avoiding multiple calculation of the same values can also appear when subintervals 
        // are further divided.


        /// <summary>Calculates an approximation of the definite interval of a tabulated function.</summary>
        /// <param name="xList">a list of independent variable. The variable can be sampled in equidistant intervals,
        /// butthis is not absolutely necessary.</param>
        /// <param name="yList">a list of corresponding function values. It must contain function values
        /// calculated at the values of the independent variable that are contained in the list a.</param>
        /// <returns></returns>
        public static double IntegralTrapezoidal(double[] xList, double[] yList)
        {
            int
                nx,  // number of points in xList
                ny,  // number of values in yList
                n; // number of subintervals
            // Argument checks:
            if (xList == null)
                throw new ArgumentNullException("The list of independent variable values is not specified (null reference).");
            if (yList == null)
                throw new ArgumentNullException("The list of function values is not specified (null reference).");
            nx = xList.Length;
            ny = yList.Length;
            if (nx < 2)
                throw new ArgumentException("Number of sampling points is less than 2 (" + nx.ToString() + ").");
            if (nx != ny)
                throw new ArgumentException("Number of sampling points (" + nx.ToString() +") and number of function values ("
                    + ny.ToString() + ") do not match.");
            n = nx - 1;
            double integral = 0.0;
            bool positivestep = (xList[1]>xList[0]); // indicates whether x values are growing
            for (int i = 0; i < n; ++i)
            {
                integral += 0.5 * (xList[i + 1] - xList[i]) * (yList[i] + yList[i + 1]);
                // Check that sampling points are ordered monotonously:
                if (positivestep && xList[i + 1] < xList[i])
                    throw new ArgumentException("Points are not ordered. Interval No. " + (i+1).ToString() 
                        + " caused exception.");
                if (!positivestep && xList[i + 1] > xList[i])
                    throw new ArgumentException("Points are not ordered. Interval No. " + (i+1).ToString() 
                        + " caused exception.");
            }
            return integral;
        }


    }
}
