﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Threading;

namespace NumLib

{

    /// <summary>Timer for measuring execution times.</summary>
    public class Timer
    {

        protected static int lastId = 0;

        public Timer()
        { Init(); }

        public Timer(string label)
        { Init(label); }

        protected void Init()
        {
            _ID = ++lastId;
            SetTimeStamp();
        }

        protected void Init(string label)
        {
            Init();
            SetLabel(label);
        }


        public static Timer Create()
        {
            Timer ret = new Timer();
            return ret;
        }

        public static Timer Create(string label)
        {
            Timer ret = new Timer(label);
            return ret;
        }

        protected object lockobj = new object();


        protected string _label = null;
        protected int _ID = 0;  


        protected double
            _creationcCpuTime, // CPU time of creation
            _totalCpuTime,  // total CPU time measured by the timer
            _startCpuTime,   // CPU time when timer was last started
            _stopCpuTime,   // CPU time when timer was last stopped
            _firstStartCpuTime;  // CPU time when timer was FIRST started

        protected DateTime
            _creationTime,    // time of creation (absolute time) 
            _startTime,     // absolute time when the timer was last started 
            _stopTime,      // absolute time when timer was last stopped 
            _firstStartTime;     // absolute time when the timer was FIRST started 

        protected double
            _totalTime;     // total wallclock time measured by the timer, in seconds 

        protected bool
            _running = false,   // tells whether timer is running or not 
            _used = false,      // tells whether timer has been used after creation or after last  reset
            _measureTime = true,     // tells whether wallclock time is measured 
            _measureCpuTime = true;  // tells whether CPU time is measured 

        #region Auxiliary

        #endregion  // Auxiliary


        /// <summary>Returns the total CPU time sent up to this moment by the current process.</summary>
        public double ThreadCpuTime()
        {
            return Process.GetCurrentProcess().TotalProcessorTime.TotalSeconds;
        }

        /// <summary>Prints out a warning related to timer use.</summary>
        /// <param name="warningstr"></param>
        protected void Warning(string warningstr)
        {
            Console.WriteLine(Environment.NewLine + Environment.NewLine
                + "Warning - timer, ID = " + ID + ": " + warningstr 
                + Environment.NewLine+Environment.NewLine);
        }

        #region Operations

        /// <summary>Starts the timer (the elapsed time & CPU time are measured since this moment). 
        /// If the timer is already running then exception is thrown.</summary>
        public void Start()
        {
            if (_running)
                Warning("Timer can not be started, since it is already running.");
            else
            {
                _startTime = DateTime.Now;
                _startCpuTime = ThreadCpuTime();
                _measureTime = _measureCpuTime = true;
                _running = true;
                if (!_used)
                {
                    _used = true;
                    _firstStartTime = _startTime;
                    _firstStartCpuTime = _startCpuTime;
                }
            }
        }

        /// <summary>Stops the timer and adds the time and CPU time difference measured in the 
        /// last round to the total time. If the timer is not running then exception is thrown.</summary>
        public void Stop()
        {
            if (!_running)
                Warning("timer can not be stopped, it is not running.");
            else
            {
                _running = false;
                if (_measureCpuTime)
                {
                    _stopCpuTime = ThreadCpuTime();
                    _totalCpuTime += _stopCpuTime - _startCpuTime;
                }
                if (_measureTime)
                {
                    _stopTime = DateTime.Now;
                    _totalTime += (_stopTime - _startTime).TotalSeconds;
                }
            }
        }

        /// <summary>Resets the timer. Its state becomes identical to the state right after creation 
        /// (as it has not been used before).</summary>
        public void Reset()
        {
            _running = _used = false;
            _totalTime = _totalCpuTime = 0.0;
            _measureTime = _measureCpuTime = false;
            SetTimeStamp();
        }

        /// <summary>Sets the timestamp on the timer (marks the current time). This is done automatically 
        /// by Reset(), Create() and constructors.</summary>
        public void SetTimeStamp()
        {
            _creationTime = DateTime.Now;
            _creationcCpuTime = ThreadCpuTime();
        }

        public void SetLabel(string label)
        {
            _label = label;
        }


        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            if (!string.IsNullOrEmpty(_label))
                sb.AppendLine("Timer \"" + _label + "\":");
            if (_running)
            {
                sb.AppendLine("Total time (intermediate): " + TotalTime +
                    " s (CPU " + TotalCpuTime + " s)");
                sb.AppendLine("Last round (intermediate): " + Time +
                    " s (CPU " + CpuTime + " s)");
            }
            else
            {
                sb.AppendLine("Total elapsed time: " + TotalTime +
                    " s (CPU " + TotalCpuTime + " s)");
                sb.AppendLine("        Last round: " + Time +
                    " s (CPU " + CpuTime + " s)");
            }
            return sb.ToString();
        }

        public string ToStringLong()
        {
            StringBuilder sb = new StringBuilder();
            if (!string.IsNullOrEmpty(_label))
                sb.AppendLine("Timer \"" + _label + "\":");
            else
                sb.AppendLine(Environment.NewLine + "Timer data:");
            sb.AppendLine(ToString());
            sb.AppendLine("First start (absolute time): ");
            sb.AppendLine("First start (CPU time): ");
            sb.AppendLine("Start of the last round (absolute time): ");
            sb.AppendLine("Start of the last round (CPU time): ");
            sb.AppendLine("Stop of the last round (absolute time): ");
            sb.AppendLine("Stop of the last round (CPU time): ");
            if (_running)
                sb.AppendLine("Timer is running.");
            else
                sb.AppendLine("Timer is not running.");
            if (_used)
                sb.AppendLine("Timer has been used already.");
            else
                sb.AppendLine("Timer has not been used yet.");
            sb.AppendLine("TimeStamp: " + TimeStamp.ToString());
            return sb.ToString();
        }

        #endregion  // Operations


        #region Properties

        /// <summary>Returns counter's ID.</summary>
        public int ID
        {  get { return _ID; }  }

        /// <summary>Gets the data from time stamp contained in the timer.</summary>
        public DateTime TimeStamp
        { get { return _creationTime; } }

        public string Label
        { get { return _label; } }

        /// <summary>Gets the total time in seconds, measured by the timer up to the current moment. 
        /// If the timer is running then the current absolute time is calculated, and difference with 
        /// the last starting time added to the total time accumulated in previous rounds.</summary>
        public double TotalTime
        {
            get
            {
                if (!_measureTime)
                    return - 1.0e-9;
                double ret = _totalTime;
                if (_running)
                {
                    ret+=(DateTime.Now - _startTime).TotalSeconds;
                }
                return ret;
            }
        }

        /// <summary>Gets the total CPU time in seconds, measured by the timer up to the current moment. 
        /// If the timer is running then the current CPU time is calculated, and difference with 
        /// the last starting CPU time added to the total CPU time accumulated in previous rounds.</summary>
        public double TotalCpuTime
        {
            get
            {
                if (!_measureCpuTime)
                    return -1.0e-9;
                double ret = _totalCpuTime;
                if (_running)
                {
                    ret += ThreadCpuTime() - _startCpuTime;
                }
                return ret;
            }
        }

        /// <summary>Returns the elapsed time measured by the timer in the last round. 
        /// If the timer  is running then the current time is calculated and its difference with 
        /// the starting time returned.</summary>
        public double Time
        {
            get
            {
                if (!_measureTime)
                    return - 1.0e-9;
                if (_running)
                    return (DateTime.Now - _startTime).TotalSeconds;
                else
                    return (_stopTime - _startTime).TotalSeconds;
            }
        }

        /// <summary>Returns the elapsed CPU time measured by the timer in the last round. 
        /// If the timer  is running then the current CPU time is calculated and its difference with 
        /// the starting time returned.</summary>
        public double CpuTime
        {
            get
            {
                if (!_measureCpuTime)
                    return -1.0e-9;
                if (_running)
                    return (ThreadCpuTime() - _startCpuTime);
                else
                    return (_stopCpuTime - _startCpuTime);
            }
        }

        #endregion  // Properties

    }  // Timer


}
