﻿#region About
// Course material: C# for numerical methods
// Copyright (c) 2009-2010, Igor Grešovnik
#endregion About


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NumLib.Classes
{


    /// <summary>Vector whose components are real numbers.</summary>
    public class Vector : Vector<double>
    {

        #region Initialization

        protected Vector(){}

        public Vector(int dimension)
        {
            Init(dimension);
        }

        public Vector(int dimension, double comp)
        {
            Init(dimension, comp);
        }

        public Vector(double[] array)
        {
            Init(array);
        }

        public Vector(Vector cv)
        {
            Init(cv);
        }

        #endregion Initialization


        #region Operations


        // Implementacija podedovanih abstraktnih metod:

        protected override double Zero
        { get { return 0.0; } }

        protected override double Add(double a, double b)
        { return a + b; }

        protected override double Subtract(double a, double b)
        { return a - b; }

        protected override double Multiply(double a, double b)
        { return a * b; }

        protected override double Divide(double a, double b)
        { return a / b; }

        protected override double Negative(double a)
        { return - a; }

        protected override double Inverse(double a)
        { return 1.0 / a; }

        protected override double Conjugate(double a)
        { return a; }


        // Overloaded operators:

        public static Vector operator +(Vector a, Vector b)
        {
            Vector ret = new Vector(a);
            ret.Add(b);
            return ret;
        }

        public static Vector operator -(Vector a, Vector b)
        {
            Vector ret = new Vector(a);
            ret.Subtract(b);
            return ret;
        }

        public static double operator *(Vector a, Vector b)
        {
            return a.RightScalarProduct(b);
        }


        public static Vector operator *(Vector a, double b)
        {
            Vector ret = new Vector(a);
            ret.Multiply(b);
            return ret;
        }

        public static Vector operator *(double b, Vector a)
        {
            Vector ret = new Vector(a);
            ret.Multiply(b);
            return ret;
        }

        #endregion  // Operations

    }  // class Vector


    /// <summary>Complex vector.</summary>
    public class ComplexVector : Vector<Complex>
    {

        #region Initialization

        // Constructors:

        protected ComplexVector(){}

        public ComplexVector(int dimension)
        {
            Init(dimension);
        }

        public ComplexVector(int dimension, Complex comp)
        {
            Init(dimension, comp);
        }

        public ComplexVector(Complex[] array)
        {
            Init(array);
        }

        public ComplexVector(ComplexVector cv)
        {
            Init(cv);
        }

        #endregion   // Initialization

        #region Operations


        // Implementacija podedovanih abstraktnih metod:

        protected override Complex Zero
        { get { return new Complex(0.0, 0.0); } }

        protected override Complex Add(Complex a, Complex b)
        { return a + b; }

        protected override Complex Subtract(Complex a, Complex b)
        { return a - b; }

        protected override Complex Multiply(Complex a, Complex b)
        { return a*b; }

        protected override Complex Divide(Complex a, Complex b)
        { return a/b; }

        protected override Complex Negative(Complex a)
        {  return Complex.Zero-a; }

        protected override Complex Inverse(Complex a)
        {  return Complex.One/a; }

        protected override Complex Conjugate(Complex a)
        { return Complex.Conjugate(a); }

 
        // Overloaded operators:

        public static ComplexVector operator +(ComplexVector a, ComplexVector b)
        {
            ComplexVector ret = new ComplexVector(a);
            ret.Add(b);
            return ret;
        }

        public static ComplexVector operator -(ComplexVector a, ComplexVector b)
        {
            ComplexVector ret = new ComplexVector(a);
            ret.Subtract(b);
            return ret;
        }

        public static ComplexVector operator *(ComplexVector a, ComplexVector b)
        {
            ComplexVector ret = new ComplexVector(a);
            ret.RightScalarProduct(b);
            return ret;
        }


        public static ComplexVector operator *(ComplexVector a, Complex b)
        {
            ComplexVector ret = new ComplexVector(a);
            ret.Multiply(b);
            return ret;
        }

        public static ComplexVector operator *(Complex b, ComplexVector a)
        {
            ComplexVector ret = new ComplexVector(a);
            ret.Multiply(b);
            return ret;
        }

        #endregion  // Operations


    }  // class ComplexVector


    public abstract class Vector<ComponentType>
    {

        #region Contructors_Initialization

        protected Vector()  // hide default constructor
        {
        }


        public Vector(int dim1, ComponentType comp)
        {
            Init(dim1, comp);
        }

        public Vector(ComponentType[] array)
        {
            Init(array);
        }

        /// <summary>Initializes a matrix with dimendions dim1 and dim2.</summary>
        /// <param name="dim1">Number of rows.</param>
        /// <param name="dim2">Number of columns.</param>
        public void Init (int dim)
        {
            _dim = dim;
            if (_dim < 1)
                throw new IndexOutOfRangeException("Can not create a vector with dimension less than 1. Specified dimension: " + dim);
            _tab = new ComponentType[_dim];
            // TODO: check this! dim1;
            for (int i=0;i<_dim;++i)
                _tab[i]=default(ComponentType);
        }

        /// <summary>Initializes a vector with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Vector dimension.</param>
        public void Init(int dim, ComponentType comp)
        {
            _dim = dim;
            if (_dim < 1)
                throw new IndexOutOfRangeException("Can not create a vector of dimension less than 1. Specified dimension: " + dim);
            _tab = new ComponentType[_dim];
            _dim = dim;
            for (int i=0;i<_dim;++i)
                _tab[i]=comp;
        }

        /// <summary>Initializes a vector with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Vector dimension.</param>
        public void Init(ComponentType[] array)
        {
            _dim = array.Count<ComponentType>(); 
            if (_dim < 1)
                throw new IndexOutOfRangeException("Can not create a vector of dimension less than 1. Specified dimension: " + _dim);
            _tab = new ComponentType[_dim];
            for (int i = 0; i < _dim; ++i)
                _tab[i] = array[i];
        }

        /// <summary>Initializes a vector with dimendion dim and sets all components to comp.</summary>
        /// <param name="dim">Vector dimension.</param>
        public void Init(Vector<ComponentType> v)
        {
            _dim = v.Dimension;
            if (_dim < 1)
                throw new IndexOutOfRangeException("Can not create a vector of dimension less than 1. Specified dimension: " + _dim);
            _tab = new ComponentType[_dim];
            for (int i = 0; i < _dim; ++i)
                _tab[i] = v[i];
        }


        public static ComponentType[] Copy(Vector<ComponentType> v)
        {
            ComponentType[] ret;
            if (v == null)
                ret = null;
            else
            {
                int dim = v.Dimension;
                ret = new ComponentType[dim];
                for (int i = 0; i < dim; ++i)
                    ret[i] = v[i];
            }
            return ret;
        }


        #endregion  // Contructors_Initialization

        #region Data

        protected int _dim;
        protected ComponentType[] _tab;

        /// <summary>Returns vector dimension.</summary>
        public int Dimension { get { return _dim; } }

        /// <summary>Synonyme for Dimension, returns vector dimension.</summary>
        public int d { get { return _dim; } }

        /// <summary>Gets or sets a specific vector componene.</summary>
        /// <param name="ind">Component index running from 0 to d-1.</param>
        /// <returns>a referance to a component for a reference type ComponentType or its value for value types.</returns>
        public virtual ComponentType this[int ind]
        {
            
            get
            {
                if (ind < 0 || ind >= _dim)
                    throw new IndexOutOfRangeException("Index " + ind.ToString() + " is out of range, should be between 0 and "
                        + (_dim-1).ToString() + ".");
                return _tab[ind];
            }
            set
            {
                if (ind < 0 || ind >= _dim)
                    throw new IndexOutOfRangeException("Index " + ind.ToString() + " is out of range, should be between 0 and "
                        + (_dim-1).ToString() + ".");
                _tab[ind] = value;
            }
        }


        #endregion  // Data


        #region Operations


        /// <summary>Redefinition of Tostring(), converts a vector to string representation.</summary>
        /// <returns></returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            int d1=_dim-1;
            sb.Append('{');
            for (int i = 0; i < d1; ++i)
            {
                sb.Append(this[i].ToString());
                sb.Append(", ");
                sb.Append(this[d1]);
                sb.Append("}");
            }
            return sb.ToString();
        }

        // Spodaj deklariramo osnovne operacije na polju, nad katerim je definiran vektor. Te operacije so
        // potrebne, da lahko definiramo seštevanje vektorjev, množenje s skalarjem, skalarni produkt itd.
        // Vendar na tem mestu ne moremo operacij tudi definirati, ker so lahko argumenti poljubnih tipov in
        // zato ni znano vnaprej, kako se dejansko izvede recimo seštevanje ali množenje dveh objektov tega tipa.
        // Da gre samo za deklaracije in ne za definicije, povemo z besedo "abstract". Razred, ki vsebue vsaj 
        // eno abstraktno metodo, mora biti tudi sam deklariran kot abstrakten. Abstraktnega razreda ne moremo
        // instanciirati, lahko pa po njem dedujejo drugi razredi. Razred, ki implementira vse metode, ki so 
        // deklarirane kot abstraktne v abstraktnem razredu, po katerem deduje, je konkreten in ga lahko 
        // instanciiramo (to pomeni, da lahko z operatorjem new() naredimo objekt danega tipa).

        protected abstract ComponentType Zero { get; }
        protected abstract ComponentType Add(ComponentType a, ComponentType b);
        protected abstract ComponentType Subtract(ComponentType a, ComponentType b);
        protected abstract ComponentType Multiply(ComponentType a, ComponentType b);
        protected abstract ComponentType Divide(ComponentType a, ComponentType b);
        protected abstract ComponentType Negative(ComponentType a);
        protected abstract ComponentType Inverse(ComponentType a);
        protected abstract ComponentType Conjugate(ComponentType a);

        /// <summary>Vector addition.</summary>
        public void Add(Vector<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Vector addition: first operand is null.");
            if (a.Dimension <= 0)
                throw new ArgumentException("Vector addition: dimension of operand is not greater thatn 0.");
            if (this.Dimension!=a.Dimension)
                throw new ArgumentException("Vector addition: dimensions of operands are not consistent.");
            for (int i = 0; i < a.Dimension; ++i)
            {
                this[i] = Add(this[i], a[i]);
            }
        }

        /// <summary>Vector subtraction.</summary>
        public void Subtract(Vector<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Vector subtraction: first operand is null.");
            if (a.Dimension <= 0)
                throw new ArgumentException("Vector subtraction: dimension of operand is not greater thatn 0.");
            if (this.Dimension!=a.Dimension)
                throw new ArgumentException("Vector subtraction: dimensions of operands are not consistent.");
            for (int i = 0; i < a.Dimension; ++i)
            {
                this[i] = Subtract(this[i], a[i]);
            }
        }

        /// <summary>Multiplication of a vector by a scalar.</summary>
        public void Multiply(ComponentType scalar)
        {
             if (this.Dimension<1)
                 throw new ArgumentException("Vector subtraction: vector dimension is less than 1.");
            for (int i = 0; i < this.Dimension; ++i)
            {
                this[i] = Multiply (scalar, this[i]);
            }
        }


        /// <summary>Right scalar product.</summary>
        public ComponentType RightScalarProduct(Vector<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Vector subtraction: first operand is null.");
            if (a.Dimension <= 0)
                throw new ArgumentException("Vector subtraction: dimension of operand is not greater thatn 0.");
            if (this.Dimension != a.Dimension)
                throw new ArgumentException("Vector subtraction: dimensions of operands are not consistent.");
            ComponentType ret = Zero;
            for (int i = 0; i < a.Dimension; ++i)
            {
                ret = Add(ret, Multiply(this[i], Conjugate(a[i])));
            }
            return ret;
        }

        /// <summary>Right scalar product.</summary>
        public ComponentType LeftScalarProduct(Vector<ComponentType> a)
        {
            if (a == null)
                throw new ArgumentNullException("Vector subtraction: first operand is null.");
            if (a.Dimension <= 0)
                throw new ArgumentException("Vector subtraction: dimension of operand is not greater thatn 0.");
            if (this.Dimension != a.Dimension)
                throw new ArgumentException("Vector subtraction: dimensions of operands are not consistent.");
            ComponentType ret = Zero;
            for (int i = 0; i < a.Dimension; ++i)
            {
                ret = Add(ret, Multiply(a[i], Conjugate(this[i])));
            }
            return ret;
        }

        #endregion // Operations

        public static void Example()
        {
            Console.WriteLine(); Console.WriteLine();
            Console.WriteLine("Example: complex vectors:");
            ComplexVector a = null, b = null;
            a = new ComplexVector(3);
            a[0] = new Complex(1, 2);
            a[1] = new Complex(4, 1);
            a[2] = new Complex(0.5, 0.6);
            b = new ComplexVector(3);
            b[0] = new Complex(5, 2);
            b[1] = new Complex(0.2, 0.4);
            b[2] = new Complex(9, 0);
            Console.WriteLine("a = " +  a.ToString());
            Console.WriteLine("b = " +  b.ToString());
            Console.WriteLine("a + b = " +  (a+b).ToString());


        }



    }  // class Vector

}
